/*
 *  							PS/2 Keyer
 *
 *  Copyright (C) 2009  David Bern, W2LNX     W2LNX@ARRL.net
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free Software Foundation, Inc.,
 *  51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301,
 *  USA, or see <http://www.gnu.org/licenses/>.
 */

#include "device.h"
#include "configuration.h"
#include "constants.h"
#include "ps2_protocol_keyboard.h"
#include "ps2_keyboard.h"
#include "main.h"
#include <string.h>
#include <assert.h>

/******************************************************************************/
/*																			  */
/*				PS/2 keyboard specific protocol routines					  */
/*																			  */
/*		ASCII characters are converted into PS/2 keyboard scancodes			  */
/*																			  */
/******************************************************************************/

/* -------------------------------------------------------------------------- */

enum {		/* keyboard host command codes */

	KEYBOARD_RESET				= 0xFF,
	KEYBOARD_RESEND				= 0xFE,	

	READ_ID_CMD					= 0xF2,

	SET_TYPEMATIC_CMD			= 0xF3,
		TYPEMATIC_VALUE1		= 0x00,
		TYPEMATIC_VALUE2		= 0x20,

	SET_DEFAULT_CMD				= 0xF6,

	SET_LED_CMD 				= 0xED,
		CAPS_LOCK_LED	 		= 0x04,
		NUM_LOCK_LED			= 0x02,
		SCROLL_LOCK_LED			= 0x01,
		CLEAR_LEDS				= 0x00,
};

enum {		/* keyboard device response codes */

	KEYBOARD_BAT			= 0xAA,
	KEYBOARD_ACK			= 0xFA,
	KEYBOARD_ID_1			= 0xAB,
		KEYBOARD_ID_2		= 0x83,
};

enum {		/* keyboard device scan codes */

	CAPS_LOCK_CODE		= 0x58,
	NUM_LOCK_CODE 		= 0x77,
	SCROLL_LOCK_CODE	= 0x7E,

	RIGHT_SHIFT_CODE	= 0x59,
	ENTER_CODE			= 0x5A,
	F10_CODE			= 0x09,

	BREAK_CODE			= 0xF0,
};

/* -------------------------------------------------------------------------- */

/*
 * ASCII to Scan Code Set 2 translation table
 *
 *	index - ASCII character
 *  entry - scancode
 */

struct Scancode {
	int shift;
	int code;
};

static const struct Scancode scancode[ ] = {

	/* ASCII	  SCANCODE */
	/* =====	  ======== */

	/*   0 NUL */ { 0, 0 },
	/*   1 SOH */ { 0, 0 },
	/*   2 STX */ { 0, 0 },
	/*   3 ETX */ { 0, 0 },
	/*   4 EOT */ { 0, 0 },
	/*   5 ENQ */ { 0, 0 },
	/*   6 ACK */ { 0, 0 },
	/*   7 BEL */ { 0, 0 },
	/*   8 BS  */ 	{ FALSE, 0x66 },	/* BKSP */
	/*   9 HT  */ 	{ FALSE, 0x0D },	/* TAB */
	/*  10 LF  */ { 0, 0 },
	/*  11 VT  */ { 0, 0 },
	/*  12 FF  */ { 0, 0 },
	/*  13 CR  */ 	{ FALSE, 0x5A },	/* ENTER */
	/*  14 SO  */ { 0, 0 },
	/*  15 SI  */ { 0, 0 },
	/*  16 DLE */ { 0, 0 },
	/*  17 DC1 */ { 0, 0 },
	/*  18 DC2 */ { 0, 0 },
	/*  19 DC3 */ { 0, 0 },
	/*  20 DC4 */ { 0, 0 },
	/*  21 NAK */ { 0, 0 },
	/*  22 SYN */ { 0, 0 },
	/*  23 ETB */ { 0, 0 },
	/*  24 CAN */ { 0, 0 },
	/*  25 EM  */ { 0, 0 },
	/*  26 SUB */ { 0, 0 },
	/*  27 ESC */ 	{ FALSE, 0x76 },	/* ESC */
	/*  28 FS  */ { 0, 0 },
	/*  29 GS  */ { 0, 0 },
	/*  30 RS  */ { 0, 0 },
	/*  31 US  */ { 0, 0 },
	/*  32 SP  */  	{ FALSE, 0x29 },	/* SPACE */
	/*  33 !   */   { TRUE,  0x16 },		/* shift 1 */
	/*  34 "   */  	{ TRUE,  0x52 }, 	/* shift ' */
	/*  35 #   */   { TRUE,  0x26 },		/* shift 3 */
	/*  36 $   */   { TRUE,  0x25 },		/* shift 4 */
	/*  37 %   */   { TRUE,  0x2E },		/* shift 5 */
	/*  38 &   */   { TRUE,  0x3D },		/* shift 7 */
	/*  39 '   */  	{ FALSE, 0x52 },
	/*  40 (   */  	{ TRUE,  0x46 }, 	/* shift 9 */
	/*  41 )   */  	{ TRUE,  0x45 },		/* shift 0 */
	/*  42 *   */   { TRUE,  0x3E },		/* shift 8 */
	/*  43 +   */  	{ TRUE,  0x55 },		/* shift = */
	/*  44 ,   */  	{ FALSE, 0x41 },
	/*  45 -   */  	{ FALSE, 0x4E },
	/*  46 .   */  	{ FALSE, 0x49 },
	/*  47 /   */  	{ FALSE, 0x4A },
	/*  48 0   */  	{ FALSE, 0x45 },
	/*  49 1   */  	{ FALSE, 0x16 },
	/*  50 2   */ 	{ FALSE, 0x1E },
	/*  51 3   */  	{ FALSE, 0x26 },
	/*  52 4   */  	{ FALSE, 0x25 },
	/*  53 5   */  	{ FALSE, 0x2E },
	/*  54 6   */  	{ FALSE, 0x36 },
	/*  55 7   */  	{ FALSE, 0x3D },
	/*  56 8   */  	{ FALSE, 0x3E },
	/*  57 9   */ 	{ FALSE, 0x46 },
	/*  58 :   */  	{ TRUE,  0x4C },		/* shift; */
	/*  59 ;   */   { FALSE, 0x4C },
	/*  60 <   */   { TRUE,  0x41 },		/* shift ,; */
	/*  61 =   */  	{ FALSE, 0x55 },
	/*  62 >   */   { TRUE,  0x49 },		/* shift .; */
	/*  63 ?   */  	{ TRUE,  0x4A },		/* shift / */
	/*  64 @   */  	{ TRUE,  0x1E },		/* shift 2 */
	/*  65 A   */  	{ TRUE,  0x1C },
	/*  66 B   */  	{ TRUE,  0x32 },
	/*  67 C   */  	{ TRUE,  0x21 },
	/*  68 D   */  	{ TRUE,  0x23 },
	/*  69 E   */  	{ TRUE,  0x24 },
	/*  70 F   */  	{ TRUE,  0x2B },
	/*  71 G   */  	{ TRUE,  0x34 },
	/*  72 H   */  	{ TRUE,  0x33 },
	/*  73 I   */  	{ TRUE,  0x43 },
	/*  74 J   */  	{ TRUE,  0x3B },
	/*  75 K   */  	{ TRUE,  0x42 },
	/*  76 L   */  	{ TRUE,  0x4B },
	/*  77 M   */  	{ TRUE,  0x3A },
	/*  78 N   */  	{ TRUE,  0x31 },
	/*  79 O   */ 	{ TRUE,  0x44 },
	/*  80 P   */  	{ TRUE,  0x4D },
	/*  81 Q   */  	{ TRUE,  0x15 },
	/*  82 R   */  	{ TRUE,  0x2D },
	/*  83 S   */  	{ TRUE,  0x1B },
	/*  84 T   */  	{ TRUE,  0x2C },
	/*  85 U   */  	{ TRUE,  0x3C },
	/*  86 V   */  	{ TRUE,  0x2A },
	/*  87 W   */  	{ TRUE,  0x1D },
	/*  88 X   */  	{ TRUE,  0x22 },
	/*  89 Y   */  	{ TRUE,  0x35 },
	/*  90 Z   */  	{ TRUE,  0x1A },
	/*  91 [   */   { FALSE, 0x54 },
	/*  92 \   */ 	{ FALSE, 0x5D },
	/*  93 ]   */ 	{ FALSE, 0x5B },
	/*  94 ^   */   { TRUE,  0x36 },		/* shift 6 */
	/*  95 _   */   { TRUE,  0x4E },		/* shift - */
	/*  96 `   */ 	{ FALSE, 0x0E },
	/*  97 a   */ 	{ FALSE, 0x1C },
	/*  98 b   */  	{ FALSE, 0x32 },
	/*  99 c   */ 	{ FALSE, 0x21 },
	/* 100 d   */ 	{ FALSE, 0x23 },
	/* 101 e   */ 	{ FALSE, 0x24 },
	/* 102 f   */	{ FALSE, 0x2B },
	/* 103 g   */ 	{ FALSE, 0x34 },
	/* 104 h   */ 	{ FALSE, 0x33 },
	/* 105 i   */ 	{ FALSE, 0x43 },
	/* 106 j   */ 	{ FALSE, 0x3B },
	/* 107 k   */ 	{ FALSE, 0x42 },
	/* 108 l   */ 	{ FALSE, 0x4B },
	/* 109 m   */ 	{ FALSE, 0x3A },
	/* 110 n   */ 	{ FALSE, 0x31 },
	/* 111 o   */ 	{ FALSE, 0x44 },
	/* 112 p   */  	{ FALSE, 0x4D },
	/* 113 q   */ 	{ FALSE, 0x15 },
	/* 114 r   */ 	{ FALSE, 0x2D },
	/* 115 s   */ 	{ FALSE, 0x1B },
	/* 116 t   */ 	{ FALSE, 0x2C },
	/* 117 u   */ 	{ FALSE, 0x3C },
	/* 118 v   */ 	{ FALSE, 0x2A },
	/* 119 w   */ 	{ FALSE, 0x1D },
	/* 120 x   */ 	{ FALSE, 0x22 },
	/* 121 y   */	{ FALSE, 0x35 },
	/* 122 z   */ 	{ FALSE, 0x1A },
	/* 123 {   */   { TRUE,  0x54 },		/* shift [ */
	/* 124 |   */ 	{ TRUE,  0x5D },		/* shift \ */
	/* 125 }   */ 	{ TRUE,  0x5B },		/* shift ] */
	/* 126 ~   */ 	{ TRUE,  0x0E },		/* shift ` */
	/* 127 DEL */ { 0, 0 }
};

/* -------------------------------------------------------------------------- */

static void initialize_device_keyboard(void);

static void initialize_host_keyboard(void);

static void send_scancode_to_host(unsigned int scancode);

/* -------------------------------------------------------------------------- */

/*
 * this routine initialize both the PS/2 device and host keyboard ports
 */

void initialize_ps2_keyboard(void)
{

#if TESTING == 1
	printf("initialize_ps2_keyboard():\r\n");
#endif
	initialize_device_keyboard();

	initialize_host_keyboard();

#if TESTING == 1
	printf("initialize_ps2_keyboard() done\r\n");
#endif
}

/* -------------------------------------------------------------------------- */

/*
 * this routine initialize the PS/2 device keyboard port
 *
 *		-- act as a host connected to a keyboard device  
 */

static void initialize_device_keyboard(void)
{

#if TESTING == 1
	printf("    initialize_device_keyboard():\r\n");
#endif

	/* inhibit communications of keyboard device */
	output_low(keyboard_device_clock);

	// STUB ROUTINE

#if TESTING == 1
	printf("    initialize_device_keyboard() done\r\n");
#endif
}

/* -------------------------------------------------------------------------- */

/*
 * this routine initialize the PS/2 host keyboard port
 *
 *		-- act as a keyboard device connected to host
 */

/*
 * note: this routine only supports plugging in a PS/2 to USB adapter into USB 
 * port after booting Windows XP
 */

static void initialize_host_keyboard(void)
{
	unsigned int commandcode;

#if TESTING == 1
	printf("    initialize_host_keyboard():\r\n");
#endif

	/* inhibit communications of keyboard host */
	output_low(keyboard_host_clock);
	delay_ms(100);

	/* send keyboard BAT success code */
	send_to_host_keyboard(KEYBOARD_BAT);

	for (;;) {
		
		/* get a command code from host */
		commandcode = receive_from_host_keyboard_wait();
		send_to_host_keyboard(KEYBOARD_ACK);

		if (commandcode == SET_LED_CMD) {

			continue;

		} else if (commandcode == CLEAR_LEDS) {

			break;

		} else if (commandcode == CAPS_LOCK_LED) {

			break;

		} else if (commandcode == SET_DEFAULT_CMD) {	/* for NUE-PSK */

			break;
		} else {

			assert(("unknown command code", FALSE));
			break;
		}

	}

#if TESTING == 1
	printf("    initialize_host_keyboard() done\r\n");
#endif
}

/* -------------------------------------------------------------------------- */

/*
 * this routine sends an ASCII character to the host
 */

void send_char_to_host(unsigned int ch)
{

	/* key press */
	if (scancode[ch].shift == TRUE) {
		send_to_host_keyboard(RIGHT_SHIFT_CODE);
	}
	send_to_host_keyboard(scancode[ch].code);
	delay_ms(50);

	/* key release */
	send_to_host_keyboard(BREAK_CODE);
	send_to_host_keyboard(scancode[ch].code);
	if (scancode[ch].shift == TRUE) {
		send_to_host_keyboard(BREAK_CODE);
		send_to_host_keyboard(RIGHT_SHIFT_CODE);
	}
	delay_ms(50);
}

/* -------------------------------------------------------------------------- */

/*
 * this routine sends a keyboard scancode to the host
 */

static void send_scancode_to_host(unsigned int scancode)
{

	send_to_host_keyboard(scancode);
	delay_ms(50);

	send_to_host_keyboard(BREAK_CODE);
	send_to_host_keyboard(scancode);
	delay_ms(50);
}

/* -------------------------------------------------------------------------- */

/*
 * this routine toggles caps lock
 */

void toggle_caps_lock_keyboard(void)
{
	unsigned int commandcode;

	send_to_host_keyboard(CAPS_LOCK_CODE);
	commandcode = receive_from_host_keyboard_wait();
	send_to_host_keyboard(KEYBOARD_ACK);
	assert (commandcode == SET_LED_CMD);

	commandcode = receive_from_host_keyboard_wait();
	send_to_host_keyboard(KEYBOARD_ACK);
	assert (commandcode == CAPS_LOCK_LED || commandcode == CLEAR_LEDS);

	send_to_host_keyboard(BREAK_CODE);
	send_to_host_keyboard(CAPS_LOCK_CODE);
}

/* -------------------------------------------------------------------------- */

/*
 * for the NUE-PSK: send the transmit/receive toggle F10 scancode
 */

void send_F10_keyboard(void)
{

	send_scancode_to_host(F10_CODE);
}

/* -------------------------------------------------------------------------- */

